<?php
session_start();
require_once __DIR__ . '/../config/db.php';

// Solo admins
if (empty($_SESSION['user']) || $_SESSION['user']['role'] !== 'admin') {
    header('Location: ../public/login.php');
    exit;
}

$id = (int)($_GET['id'] ?? 0);
$stmt = $pdo->prepare("SELECT * FROM products WHERE id=:id");
$stmt->execute([':id'=>$id]);
$product = $stmt->fetch();

if (!$product) {
    echo "<div class='alert alert-danger'>Producto no encontrado</div>";
    exit;
}

$errors = [];
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = trim($_POST['name'] ?? '');
    $desc = trim($_POST['description'] ?? '');
    $price = (float)($_POST['price'] ?? 0);
    $category_id = (int)($_POST['category_id'] ?? 0);

    if (!$name || $price <= 0) $errors[] = "Nombre y precio son obligatorios.";

    // Procesar imágenes subidas
    if (!empty($_FILES['images']['name'][0])) {
        $uploadDir = __DIR__ . '/../assets/img/';
        $uploadedImages = [];
        foreach ($_FILES['images']['name'] as $key => $filename) {
            $tmpName = $_FILES['images']['tmp_name'][$key];
            $ext = pathinfo($filename, PATHINFO_EXTENSION);
            $allowed = ['jpg','jpeg','png','gif','webp'];
            if (!in_array(strtolower($ext), $allowed)) {
                $errors[] = "Solo se permiten imágenes (jpg, png, gif, webp).";
                continue;
            }
            $newName = uniqid() . '.' . $ext;
            if (move_uploaded_file($tmpName, $uploadDir . $newName)) {
                $uploadedImages[] = $newName;
            }
        }

        // Borrar imágenes anteriores si se subieron nuevas
        if ($uploadedImages) {
            $stmt = $pdo->prepare("DELETE FROM product_images WHERE product_id=:pid");
            $stmt->execute([':pid'=>$id]);

            foreach ($uploadedImages as $img) {
                $stmt = $pdo->prepare("INSERT INTO product_images (product_id,image) VALUES (:pid,:img)");
                $stmt->execute([':pid'=>$id, ':img'=>$img]);
            }
        }
    }

    if (!$errors) {
        $stmt = $pdo->prepare("UPDATE products SET name=:name, description=:desc, price=:price, category_id=:cat WHERE id=:id");
        $stmt->execute([
            ':name'=>$name,
            ':desc'=>$desc,
            ':price'=>$price,
            ':cat'=>$category_id>0?$category_id:null,
            ':id'=>$id
        ]);
        $success = "✅ Producto actualizado correctamente.";
    }
}

// Traer categorías
$categories = $pdo->query("SELECT id, name FROM categories ORDER BY name")->fetchAll();

// Traer imágenes actuales
$stmtImgs = $pdo->prepare("SELECT image FROM product_images WHERE product_id=:pid");
$stmtImgs->execute([':pid'=>$id]);
$images = $stmtImgs->fetchAll(PDO::FETCH_COLUMN);
?>

<?php if($errors): ?>
    <?php foreach($errors as $err): ?>
        <div class="alert alert-danger"><?= htmlspecialchars($err) ?></div>
    <?php endforeach; ?>
<?php endif; ?>

<?php if($success): ?>
    <div class="alert alert-success"><?= $success ?></div>
<?php endif; ?>

<form method="post" enctype="multipart/form-data">
    <div class="mb-3">
        <label>Nombre</label>
        <input type="text" name="name" class="form-control" value="<?= htmlspecialchars($product['name']) ?>" required>
    </div>

    <div class="mb-3">
        <label>Descripción</label>
        <textarea name="description" class="form-control"><?= htmlspecialchars($product['description']) ?></textarea>
    </div>

    <div class="mb-3">
        <label>Precio</label>
        <input type="number" step="0.01" name="price" class="form-control" value="<?= htmlspecialchars($product['price']) ?>" required>
    </div>

    <div class="mb-3">
        <label>Categoría</label>
        <select name="category_id" class="form-select">
            <option value="0">Sin categoría</option>
            <?php foreach($categories as $c): ?>
                <option value="<?= (int)$c['id'] ?>" <?= $c['id']==$product['category_id']?'selected':'' ?>><?= htmlspecialchars($c['name']) ?></option>
            <?php endforeach; ?>
        </select>
    </div>

    <div class="mb-3">
        <label>Imágenes</label>
        <input type="file" name="images[]" class="form-control" multiple accept="image/*">
        <?php if($images): ?>
            <div class="mt-2 d-flex flex-wrap gap-2">
                <?php foreach($images as $img): ?>
                    <img src="<?= BASE_URL . '/../assets/img/' . $img ?>" style="width:80px;height:80px;object-fit:cover;">
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>

    <button class="btn btn-primary">Guardar cambios</button>
</form>
